/**
 * 
 */
package psr.xml.utility;

import java.io.File;
import java.io.FileInputStream;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;

import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.io.SAXReader;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.XMLReader;

/**
 * Simple Java class validates input XML document using the schedule file
 * provided in command line.
 * 
 * <p>
 * Copyright 2007 Accenture LLP. This document is licensed to CalPERS only for
 * CalPERS program purposes.
 * </p>
 * 
 * @author Accenture Technical Architecture Team
 */
public class SimpleXmlValidator extends SAXReader
{
    private static final String schemaLanguage = "http://www.w3.org/2001/XMLSchema";
    
    private Schema schema;
    
    /**
     * Constructs a new instance of XmlReader class.
     *
     */
    public SimpleXmlValidator()
    {
        this.setErrorHandler(new ExceptionErrorHandler());
    }
    
    /**
     * {@inheritDoc} 
     *
     * @see org.dom4j.io.SAXReader#createXMLReader()
     */
    protected XMLReader createXMLReader() throws SAXException
    {
        try
        {
            SAXParserFactory factory = SAXParserFactory.newInstance();
            if (schema != null)
                factory.setSchema(schema);
            
            SAXParser parser = factory.newSAXParser();
            return  parser.getXMLReader();
        }
        catch (ParserConfigurationException e)
        {
            throw new SAXException(e);
        }
    }
    
    /**
     * {@inheritDoc} 
     *
     * @see org.dom4j.io.SAXReader#read(org.xml.sax.InputSource)
     */
    public Document read(InputSource source) 
        throws DocumentException, XmlValidationException
    {
        try
        {
            return super.read(source);
        }
        catch (DocumentException e)
        {
            if(e.getNestedException() != null && 
                    e.getNestedException() instanceof XmlValidationException)
            {
                throw (XmlValidationException) e.getNestedException();
            }
            else
            {
                throw e;
            }
        }
    }

    /**
     * Setter for the schema as a set of sources.
     * 
     * @param schemas
     *            The array list of the schema sources.
     * 
     * @throws SAXException
     *             This exception is thrown when there are issues creating
     *             schema from the schema sources that are passed in.
     */
    public void setSchema(Source[] schemas) throws SAXException
    {
        this.schema = SchemaFactory.newInstance(schemaLanguage).newSchema(
                schemas);
    }
    
    /**
     * Main method for executing XML validation 
     *
     * @param args the command line arguments
     */
    public static void main(String[] args)
    {
        if (args.length < 4)
        {
            System.err.println("#######################################################################");            
            System.err.println("#                                                                     #");
            System.err.println("# Usage: xmlvalidator <Xml Data File> <Xsd File>                      #");
            System.err.println("#                                                                     #");
            System.err.println("# Example:                                                            #");
            System.err.println("#        xmlvalidator instance.xml RetirementHealthTransactionsV1.xsd #");
            System.err.println("#                                                                     #");            
            System.err.println("#######################################################################");            
            System.err.println("");
            System.exit(0);
        }
        
        try
        {
            Source[] schemas = new Source[args.length - 1];
            int idx = 0;
            schemas[idx++] = new StreamSource(new FileInputStream(new File(args[args.length - 2])));
            for (int i = 1; i <= args.length - 3; i++)
            {
                schemas[idx++] = new StreamSource(new FileInputStream(new File(args[i])));        
            }
            schemas[idx] = new StreamSource(new FileInputStream(new File(args[args.length - 1])));
                               
            SimpleXmlValidator xmlValidator = new SimpleXmlValidator();
            xmlValidator.setSchema(schemas);
            
            xmlValidator.read(new File(args[0]));
            System.out.println("");
            System.out.println("Succeeded in XML schema validation for '" + args[0] + "'");
            System.out.println("");
        }
        catch (Exception e)
        {
            System.err.println();
            System.err.println("XML file '" + args[0] + "'failed in schema validation");
            e.printStackTrace();
        }
    }
    
    private class ExceptionErrorHandler implements ErrorHandler
    {
        /**
         * {@inheritDoc} 
         *
         * @see org.xml.sax.ErrorHandler#error(org.xml.sax.SAXParseException)
         */
        public void error(SAXParseException exception) throws SAXException
        {
            throw new XmlValidationException("error while parsing XML.", exception);        
        }
        
        /**
         * {@inheritDoc} 
         *
         * @see org.xml.sax.ErrorHandler#fatalError(org.xml.sax.SAXParseException)
         */
        public void fatalError(SAXParseException exception) throws SAXException
        {
            throw new XmlValidationException("fatal error while parsing XML.", exception);        
        }
        
        /**
         * {@inheritDoc} 
         *
         * @see org.xml.sax.ErrorHandler#warning(org.xml.sax.SAXParseException)
         */
        public void warning(SAXParseException exception) throws SAXException
        {
            throw new XmlValidationException("warning while parsing XML.", exception);        
        }    
    }
    
    private class XmlValidationException extends RuntimeException
    {
        /**
         *  Creates an instance of this exception with a message
         *  and the initial cause.
         *  
         * @param message Exception message.
         * @param e Original cause of the exception.
         */
        public XmlValidationException(String message, Exception e)
        {
            super(e);
        }
    }
}
